/**
  *******************************************************************************
  * 
  * @file    hal_adc.c
  * @brief   ES7P203 adc module driver.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The Adc driver can be used as follows:
    (+) Initialize the Adc registers in slave mode by using adc_init().
    (+) Get adc convert value adc_get_value()

  @endverbatim
  *********************************************************************************
  */
#include "hal_adc.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup ADC adc
  * @{
  */

/**
  * @defgroup Adc_Public_Functions adc Public Functions
  * @brief    ADC Initialization and Sample Transfer Functions
@verbatim
====================================================================================
  			##### ADC Public Functions #####
====================================================================================
  [..]  The section provide functions as follows:
    (+) Initialize ADC module by specific structure by adc_init().
    (+) get the adc sample transfer value.

@endverbatim
  * @{
  */

/**
  * @brief  Initialization the adc module by specific structure.
  * @param  init: specific init structure.
  * @retval status, see @ref hal_status_t.
  */
hal_status_t adc_init(adc_init_t *init)
{
	if (init == NULL)
		return ERROR;

	assert_param(IS_ADC_SPEED_TYPE(init->speed));
	assert_param(IS_ADC_SAMP_CLK_TYPE(init->samp_clk));
	assert_param(IS_ADC_CLK_DIV_TYPE(init->clk_div));
	assert_param(IS_ADC_ALIGN_TYPE(init->align));
	assert_param(IS_ADC_START_TYPE(init->start_mode));
	assert_param(IS_ADC_CHANNLE_TYPE(init->channel));
	
	if (init->channel <= ADC_CHANNEL_3) {
		PET |= 1 << (4 + init->channel);
		PES |= 1 << (4 + init->channel);
	}
	else {
		PBT |= 1 << (init->channel - 4);
		PBS |= 1 << (init->channel - 4);
	}

	MODIFY_REG(ADCCH, (0x7 << 4), (init->clk_div << 4));
	MODIFY_REG(ADCCH, (0x3 << 2), (init->samp_clk << 2));
	ADFM  = init->align;

	if (init->speed == ADC_HIGH_SPEED)
		ADVCMHS = 1;
	else
		ADVCMHS = 0;
	
	ADHSEN = init->speed;
	SMPS   = init->start_mode;
	MODIFY_REG(ADCCL, (0xF << 4), (init->channel << 4));
	ADEN = 1;
	return OK;
}

/**
  * @brief  get adc sample transfer value.
  * @retval data: transfer value.
  */
uint16_t adc_get_value(void)
{	
	while (ADTRG);
	return ADCRH<<8 | ADCRL;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
